/****************************************************************************
 *  Project: NXP LPC8xx SCT reload example
 *	Simple SCT program Use the match reload registers to change the duty cycle of two PWM
 *	signals and maintain their dead-time intervals using the NORELOAD_L bit in the
 *	SCT Configuration register

****************************************************************************/

#include "board.h"

#define DC1 	(130)
#define DC2 	(135)
#define Period 	(180)

void sct_fsm_init (void);


void sct_fsm_init (void)
{

	Chip_SCT_Init(LPC_SCT);			                  		 			// enable the SCT clock

	Chip_SCT_Config(LPC_SCT,  SCT_CONFIG_16BIT_COUNTER |
                            SCT_CONFIG_AUTOLIMIT_L    );					// split timers


	Chip_SCT_SetControl(LPC_SCT, SCT_CTRL_BIDIR_L(1));     				// configure SCT as BIDIR

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_0, Period);					// match on (half) PWM period
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_0, Period);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_1, DC1);					// match on duty cycle 1
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_1, DC1);

	Chip_SCT_SetMatchCount(LPC_SCT, SCT_MATCH_2, DC2);				// match on duty cycle 2
	Chip_SCT_SetMatchReload(LPC_SCT, SCT_MATCH_2, DC2);

	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_0, ENABLE_ALL_STATES);	 	// event 0 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_0, (CHIP_SCT_EVENTCTRL_T)  ( SCT_EVECTRL_MATCH1 | 
                                                                        SCT_COMBMODE_MATCH ));	// match 1 (DC1) only condition


	Chip_SCT_EventState(LPC_SCT, SCT_EVENT_1, ENABLE_ALL_STATES);	 	    // event 1 happens in all states
	Chip_SCT_EventControl(LPC_SCT, SCT_EVENT_1, (CHIP_SCT_EVENTCTRL_T)  (  SCT_EVECTRL_MATCH2	|
                                                                          SCT_COMBMODE_MATCH	));  // match 2 (DC) only condition

	

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_0);	          // event 0 sets   OUT0 (blue LED)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_0, SCT_EVT_0);				  // event 0 clears OUT0 (blue LED)

	Chip_SCT_SetOutput(LPC_SCT, SCT_OUTPUT_1, SCT_EVT_1);				    // event 1 sets   OUT1 (red LED)
	Chip_SCT_ClearOutput(LPC_SCT, SCT_OUTPUT_1,SCT_EVT_1); 	        // event 1 clears OUT1 (red LED)

	Chip_SCT_SetConflictResolution(LPC_SCT, 0,SCT_RES_TOGGLE_OUTPUT);  	// toggle OUT0 and OUT1 on conflict

	Chip_SCT_Output(LPC_SCT, 0x01);                   					// default set OUT0 and and clear OUT1


	Chip_SCT_ClearControl(LPC_SCT,SCT_CTRL_HALT_L);	   					// unhalt it by clearing bit 2 of the CTRL register

}



void SysTick_Handler(void)
{

	Chip_SCT_SetControl(LPC_SCT, SCT_CONFIG_NORELOADL_U	); 			/* stop reload process for L and H counters */

	if (Chip_GPIO_GetPinState(LPC_GPIO_PORT, 0, 0) & 0x1)			/* PIO0_0 high ?  */
	{
		if (LPC_SCT->MATCHREL_L[2] < Period-1)						// check if DC2 < Period of PWM
		{
			LPC_SCT->MATCHREL_L[1]++;
			LPC_SCT->MATCHREL_L[2]++;
		}
	}
	else
	{
		if (LPC_SCT->MATCHREL_L[1]>1)
		{
			LPC_SCT->MATCHREL_L[1]--;
			LPC_SCT->MATCHREL_L[2]--;
		}
	}

	Chip_SCT_ClearControl(LPC_SCT, SCT_CONFIG_NORELOADL_U);				/* enable reload process for L and H counters */
}


int main(void)
{
	/* Generic Initialization */
	SystemCoreClockUpdate();
	Board_Init();


	Chip_SWM_Init();						 							/* Configure the switch matrix                           */
  
  Chip_SWM_DisableFixedPin(SWM_FIXED_ACMP_I1);                    //disable Pinmux for P0_0 (ACMP_I1)
	
	Chip_SWM_MovablePinAssign(SWM_CTOUT_0_O, 16);						// SCT_OUT0 at P0_16
	Chip_SWM_MovablePinAssign(SWM_CTOUT_1_O, 7);						// SCT_OUT1 at P0_7

  SysTick_Config(SystemCoreClock/50);   								/* SysTick interrupt every 100 ms                       */
  
  NVIC_EnableIRQ(SysTick_IRQn);         								/* enable the Systick interrupt */

	sct_fsm_init();                      							    /* Initialize the SCT using the code generated by Red State */

    while(1)
    {
//    	__WFI();
    }
}
